<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

require_once "../config/conexion.php";

// Manejar el marcado como devuelto
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['id_prestamo_devuelto'])) {
    $id_prestamo = (int)$_POST['id_prestamo_devuelto'];
    $fecha_actual = date('Y-m-d');

    // 1. Actualizar fecha_devolucion_real en prestamos
    $update_prestamo = $conn->prepare("UPDATE prestamos SET fecha_devolucion_real = ? WHERE id = ?");
    $update_prestamo->bind_param("si", $fecha_actual, $id_prestamo);
    $update_prestamo->execute();
    $update_prestamo->close();

    // 2. Obtener libro_id para actualizar stock_disponible
    $get_libro = $conn->prepare("SELECT libro_id FROM prestamos WHERE id = ?");
    $get_libro->bind_param("i", $id_prestamo);
    $get_libro->execute();
    $result_libro = $get_libro->get_result();
    $libro_id = null;
    if ($row = $result_libro->fetch_assoc()) {
        $libro_id = $row['libro_id'];
    }
    $get_libro->close();

    if ($libro_id) {
        // 3. Actualizar stock_disponible del libro sumando 1
        $update_libro = $conn->prepare("UPDATE libros SET stock_disponible = stock_disponible + 1 WHERE id = ?");
        $update_libro->bind_param("i", $libro_id);
        $update_libro->execute();
        $update_libro->close();
    }

    // Redirigir para evitar reenvío del formulario
    header("Location: historial.php");
    exit();
}

// Redirigir según rol
if (isset($_SESSION['id_rol'])) {
    switch ($_SESSION['id_rol']) {
        case 1: // Admin
            $inicio_url = '/index.php';
            break;
        case 2: // Funcionario
            $inicio_url = '/views/dashboard_funcionario.php';
            break;
        default:
            $inicio_url = '/index.php';
            break;
    }
} else {
    $inicio_url = '/index.php';
}

// Parámetros de búsqueda y paginación
$buscar = isset($_GET['buscar']) ? $_GET['buscar'] : '';
$buscar_param = "%$buscar%";
$pagina = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$por_pagina = 10;
$offset = ($pagina - 1) * $por_pagina;

// Consulta para contar total de registros (alumnos + docentes)
$sql_total = "
    SELECT COUNT(*) AS total FROM prestamos
    LEFT JOIN alumnos ON prestamos.usuario_id = alumnos.rut
    LEFT JOIN cursos ON alumnos.curso_id = cursos.id
    LEFT JOIN docentes ON prestamos.usuario_id = docentes.rut
    WHERE (alumnos.nombre_alumno COLLATE utf8mb4_general_ci LIKE ? 
       OR CONCAT(cursos.nivel COLLATE utf8mb4_general_ci, cursos.seccion COLLATE utf8mb4_general_ci) LIKE ?)
       OR (docentes.nombre_docente COLLATE utf8mb4_general_ci LIKE ?)
";

$stmt_total = $conn->prepare($sql_total);
if (!$stmt_total) {
    die("Error al preparar la consulta total: " . $conn->error);
}
$stmt_total->bind_param("sss", $buscar_param, $buscar_param, $buscar_param);
$stmt_total->execute();
$resultado_total = $stmt_total->get_result();
$total_filas = $resultado_total->fetch_assoc()['total'];
$total_paginas = ceil($total_filas / $por_pagina);

// Consulta principal (alumnos + docentes)
$sql = "
    SELECT prestamos.*, 
           alumnos.nombre_alumno AS nombre_alumno, 
           CONCAT(cursos.nivel, cursos.seccion) AS curso,
           docentes.nombre_docente AS nombre_docente,
           libros.titulo AS nombre_libro
    FROM prestamos
    LEFT JOIN alumnos ON prestamos.usuario_id = alumnos.rut
    LEFT JOIN cursos ON alumnos.curso_id = cursos.id
    LEFT JOIN docentes ON prestamos.usuario_id = docentes.rut
    LEFT JOIN libros ON prestamos.libro_id = libros.id
    WHERE (alumnos.nombre_alumno COLLATE utf8mb4_general_ci LIKE ? 
       OR CONCAT(cursos.nivel COLLATE utf8mb4_general_ci, cursos.seccion COLLATE utf8mb4_general_ci) LIKE ?)
       OR (docentes.nombre_docente COLLATE utf8mb4_general_ci LIKE ?)
    ORDER BY prestamos.fecha_prestamo DESC
    LIMIT ? OFFSET ?
";

$stmt = $conn->prepare($sql);
if (!$stmt) {
    die("Error al preparar la consulta principal: " . $conn->error);
}
$stmt->bind_param("sssii", $buscar_param, $buscar_param, $buscar_param, $por_pagina, $offset);
$stmt->execute();
$resultado = $stmt->get_result();
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Historial de Préstamos</title>
    <link rel="stylesheet" href="/css/historial.css">
</head>
<body>
    <div class="contenedor">
        <img src="/img/INSIGNIA SF.png" alt="Logo" class="logo">
        <h1>Historial de Préstamos</h1>

        <form method="GET">
            <input type="text" name="buscar" placeholder="Buscar por nombre o curso" value="<?php echo htmlspecialchars($buscar); ?>">
            <button type="submit">Buscar</button>
        </form>

        <table>
            <thead>
                <tr>
                    <th>Nombre</th>
                    <th>Curso</th>
                    <th>Libro</th>
                    <th>Fecha de Préstamo</th>
                    <th>Fecha de Devolución</th>
                    <th>Estado</th>
                    <th>Acción</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($resultado->num_rows > 0): ?>
                    <?php while ($fila = $resultado->fetch_assoc()): ?>
                        <tr>
                            <td data-label="Nombre">
                                <?php 
                                if ($fila['nombre_alumno']) {
                                    echo htmlspecialchars($fila['nombre_alumno']);
                                } elseif ($fila['nombre_docente']) {
                                    echo htmlspecialchars($fila['nombre_docente']);
                                } else {
                                    echo "Desconocido";
                                }
                                ?>
                            </td>
                            <td data-label="Curso"><?php echo htmlspecialchars($fila['curso'] ?? '-'); ?></td>
                            <td data-label="Libro"><?php echo htmlspecialchars($fila['nombre_libro']); ?></td>
                            <td data-label="Fecha de Préstamo"><?php echo htmlspecialchars($fila['fecha_prestamo']); ?></td>
                            <td data-label="Fecha de Devolución"><?php echo htmlspecialchars($fila['fecha_devolucion']); ?></td>
                            <td data-label="Estado">
                                <?php 
                                if (is_null($fila['fecha_devolucion_real']) || $fila['fecha_devolucion_real'] === '' || $fila['fecha_devolucion_real'] === '0000-00-00') {
                                    echo '<span class="estado-prestado">Prestado</span>';
                                } else {
                                    echo '<span class="estado-devuelto">Devuelto</span>';
                                }
                                ?>
                            </td>
                            <td data-label="Acción">
                                <?php if (is_null($fila['fecha_devolucion_real']) || $fila['fecha_devolucion_real'] === '' || $fila['fecha_devolucion_real'] === '0000-00-00'): ?>
                                    <form method="POST" style="margin:0;">
                                        <input type="hidden" name="id_prestamo_devuelto" value="<?php echo $fila['id']; ?>">
                                        <button type="submit" onclick="return confirm('¿Marcar este libro como devuelto?');">Marcar como Devuelto</button>
                                    </form>
                                <?php else: ?>
                                    <em>Devuelto</em>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                <?php else: ?>
                    <tr><td colspan="7">No se encontraron registros.</td></tr>
                <?php endif; ?>
            </tbody>
        </table>

        <!-- Navegación de páginas -->
        <div class="paginacion">
            <?php for ($i = 1; $i <= $total_paginas; $i++): ?>
                <a href="?buscar=<?php echo urlencode($buscar); ?>&pagina=<?php echo $i; ?>" class="<?php echo $i == $pagina ? 'activa' : ''; ?>">
                    <?php echo $i; ?>
                </a>
            <?php endfor; ?>
        </div>

        <!-- Botón volver -->
        <div class="volver-container">
            <a href="<?php echo $inicio_url; ?>" class="btn-volver">🏠 Volver al Inicio</a>
        </div>
    </div>
</body>
</html>