<?php
header('Content-Type: text/html; charset=utf-8');
session_start();
include '../includes/db.php';
include '../includes/auth.php';
checkAuth();
if (!isAdmin()) {
    header("Location: ../index.php");
    exit();
}
error_reporting(E_ALL);
ini_set('display_errors', 1);
$search_query = "";
$search_param = [];
$message = '';
$message_type = '';
// Función para validar el RUT
function validarRut($rut) {
    $rut = preg_replace('/[^0-9kK]/', '', $rut);

    if (strlen($rut) < 2) {
        return false;
    }
    $dv = strtoupper(substr($rut, -1));
    $numero = substr($rut, 0, strlen($rut) - 1);
    $i = 2;
    $suma = 0;
    foreach (array_reverse(str_split($numero)) as $n) {
        if (!is_numeric($n)) {
            return false;
        }
        $suma += $n * $i;
        $i = ($i == 7) ? 2 : $i + 1;
    }
    $dvr = 11 - ($suma % 11);
    if ($dvr == 11) {
        $dvr = '0';
    }
    if ($dvr == 10) {
        $dvr = 'K';
    }
    return ($dvr == $dv);
}
// Manejo de búsqueda
if ($_SERVER['REQUEST_METHOD'] == 'GET' && isset($_GET['buscar'])) {
    $search = trim($_GET['search']);
    if ($search !== '') {
        $search_query = "WHERE rut LIKE ? OR nombre LIKE ? OR apellido_paterno LIKE ? OR apellido_materno LIKE ? OR curso_id IN (SELECT id FROM curso WHERE nombre LIKE ?) OR rol_id IN (SELECT id FROM roles WHERE nombre LIKE ?)";
        $search_param = ["%$search%", "%$search%", "%$search%", "%$search%", "%$search%", "%$search%"];
    }
}
// Manejo de creación y modificación
if ($_SERVER['REQUEST_METHOD'] == 'POST' && (isset($_POST['crear']) || isset($_POST['modificar']))) {
    // Recoger y sanitizar datos
    $rut = filter_input(INPUT_POST, 'rut', FILTER_SANITIZE_SPECIAL_CHARS);
    $nombre = filter_input(INPUT_POST, 'nombre', FILTER_SANITIZE_SPECIAL_CHARS);
    $apellido_paterno = filter_input(INPUT_POST, 'apellido_paterno', FILTER_SANITIZE_SPECIAL_CHARS);
    $apellido_materno = filter_input(INPUT_POST, 'apellido_materno', FILTER_SANITIZE_SPECIAL_CHARS);
    $contrasena = $_POST['contrasena'];
    $rol_id = filter_input(INPUT_POST, 'rol', FILTER_VALIDATE_INT);
    $curso_id = isset($_POST['curso_id']) && $_POST['curso_id'] !== '' ? filter_input(INPUT_POST, 'curso_id', FILTER_VALIDATE_INT) : NULL;
    $activo = isset($_POST['activo']) ? 1 : 0; // Agregado campo activo
    // Validar campos vacíos o con solo espacios en blanco
    if (trim($rut) === '' || trim($nombre) === '' || trim($apellido_paterno) === '' || trim($apellido_materno) === '') {
        $message = "Error: Todos los campos deben contener texto válido.";
        $message_type = 'error';
    } else {
        // Validar RUT
        if (!validarRut($rut)) {
            $message = "Error: El RUT ingresado no es válido.";
            $message_type = 'error';
        } else {
            // Verificar si el RUT ya existe (solo para creación)
            $stmt = $conn->prepare("SELECT id FROM usuario WHERE rut = ?");
            $stmt->bind_param("s", $rut);
            $stmt->execute();
            $stmt->store_result();

            if ($stmt->num_rows > 0 && !isset($_POST['modificar'])) {
                $message = "Error: El RUT ya está registrado.";
                $message_type = 'error';
            } else {
                $stmt->close();
                $contrasena_hash = null;

                // Manejar contraseña
                if (!empty($contrasena)) {
                    if (strlen($contrasena) < 8) {
                        $message = "Error: La contraseña debe tener al menos 8 caracteres.";
                        $message_type = 'error';
                    } else {
                        $contrasena_hash = password_hash($contrasena, PASSWORD_BCRYPT);
                    }
                } elseif (isset($_POST['crear'])) {
                    $message = "Error: La contraseña es requerida para crear un usuario.";
                    $message_type = 'error';
                }
                if (empty($message)) {
                    if (isset($_POST['crear'])) {
                        $stmt = $conn->prepare("INSERT INTO usuario (rut, nombre, apellido_paterno, apellido_materno, contraseña, rol_id, curso_id, activo) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                        $stmt->bind_param("ssssssii", $rut, $nombre, $apellido_paterno, $apellido_materno, $contrasena_hash, $rol_id, $curso_id, $activo);
                        $action = 'creado';
                    } else if (isset($_POST['modificar'])) {
                        $id = filter_input(INPUT_POST, 'usuario_id', FILTER_VALIDATE_INT);
                        if ($contrasena_hash) {
                            $stmt = $conn->prepare("UPDATE usuario SET rut=?, nombre=?, apellido_paterno=?, apellido_materno=?, contraseña=?, rol_id=?, curso_id=?, activo=? WHERE id=?");
                            $stmt->bind_param("ssssssiii", $rut, $nombre, $apellido_paterno, $apellido_materno, $contrasena_hash, $rol_id, $curso_id, $activo, $id);
                        } else {
                            $stmt = $conn->prepare("UPDATE usuario SET rut=?, nombre=?, apellido_paterno=?, apellido_materno=?, rol_id=?, curso_id=?, activo=? WHERE id=?");
                            $stmt->bind_param("ssssiisi", $rut, $nombre, $apellido_paterno, $apellido_materno, $rol_id, $curso_id, $activo, $id);
                        }
                        $action = 'modificado';
                    }
                    if ($stmt->execute()) {
                        $message = "Usuario $action exitosamente.";
                        $message_type = 'exito';
                    } else {
                        $message = "Error: " . $stmt->error;
                        $message_type = 'error';
                    }
                    $stmt->close();
                }
            }
        }
    }
}
// 1. ELIMINACIÓN MÚLTIPLE (BORRADO DEFINITIVO)
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['eliminar_varios'])) {
    $ids = isset($_POST['usuario_ids']) ? $_POST['usuario_ids'] : [];

    if (!empty($ids)) {
        $ids = array_filter($ids, fn($id) => filter_var($id, FILTER_VALIDATE_INT));
        if (!empty($ids)) {
            $placeholders = implode(',', array_fill(0, count($ids), '?'));
            $types = str_repeat('i', count($ids));
            $stmt = $conn->prepare("DELETE FROM usuario WHERE id IN ($placeholders)");
            if ($stmt) {
                $stmt->bind_param($types, ...array_map('intval', $ids));
                if ($stmt->execute()) {
                    $message = "Usuarios eliminados exitosamente.";
                    $message_type = 'exito';
                } else {
                    $message = "Error al eliminar usuarios: " . $stmt->error;
                    $message_type = 'error';
                }
                $stmt->close();
            } else {
                $message = "Error al preparar la consulta de eliminación.";
                $message_type = 'error';
            }
        }
    } else {
        $message = "No se seleccionaron usuarios para eliminar.";
        $message_type = 'error';
    }
}
// 2. CAMBIO DE ESTADO A INACTIVO (ELIMINACIÓN LÓGICA INDIVIDUAL)
elseif ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['marcar_inactivo'])) {
    $id = filter_input(INPUT_POST, 'usuario_id', FILTER_VALIDATE_INT);
    if ($id) {
        $stmt = $conn->prepare("UPDATE usuario SET activo = 0 WHERE id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $id);
            if ($stmt->execute()) {
                $message = "Usuario marcado como inactivo exitosamente.";
                $message_type = 'exito';
            } else {
                $message = "Error al marcar como inactivo: " . $stmt->error;
                $message_type = 'error';
            }
            $stmt->close();
        }
    } else {
        $message = "ID de usuario inválido.";
        $message_type = 'error';
    }
}
// Consulta de usuarios
if ($search_query) {
    $stmt = $conn->prepare("SELECT * FROM usuario $search_query");
    $bind_names[] = str_repeat('s', count($search_param));
    foreach ($search_param as $key => $value) {
        $bind_name = 'bind' . $key;
        $$bind_name = $value;
        $bind_names[] = &$$bind_name;
    }
    call_user_func_array([$stmt, 'bind_param'], $bind_names);
    $stmt->execute();
    $usuarios = $stmt->get_result();
    $stmt->close();
} else {
    // Solo mostrar usuarios activos
    $usuarios = $conn->query("SELECT * FROM usuario");
}
// Parámetros de paginación
// Configuración de paginación
$registros_por_pagina = 20; // Define la cantidad de registros por página
// Obtiene la página actual desde la URL, por defecto es 1
$pagina_actual = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
// Calcula el desplazamiento para la consulta SQL
$offset = ($pagina_actual - 1) * $registros_por_pagina;
// Contar el número total de usuarios activos
$result_total_usuarios = $conn->query("SELECT COUNT(*) AS total FROM usuario WHERE activo = 1");
$total_usuarios = $result_total_usuarios->fetch_assoc()['total'];
$total_paginas = ceil($total_usuarios / $registros_por_pagina); // Total de páginas
// Consulta de usuarios
if ($search_query) {
    // Consulta de búsqueda con LIMIT y OFFSET
    $query = "SELECT * FROM usuario $search_query LIMIT ? OFFSET ?";
    $stmt = $conn->prepare($query);

    // Los tipos de parámetros deben coincidir con los parámetros de búsqueda más LIMIT y OFFSET
    $bind_types = str_repeat('s', count($search_param)) . "ii"; // 's' para strings, 'ii' para LIMIT y OFFSET
    $params = array_merge($search_param, [$registros_por_pagina, $offset]);

    // Llamar a bind_param con un array de parámetros dinámicos
    $stmt->bind_param($bind_types, ...$params);

    // Ejecutar y obtener el resultado
    $stmt->execute();
    $usuarios = $stmt->get_result();
    $stmt->close();
} else {
    // Consulta sin búsqueda, pero con paginación
    $stmt_usuarios = $conn->prepare("SELECT * FROM usuario WHERE activo = 1 LIMIT ? OFFSET ?");
    $stmt_usuarios->bind_param("ii", $registros_por_pagina, $offset);
    $stmt_usuarios->execute();
    $usuarios = $stmt_usuarios->get_result();
}
// Obtener cursos y roles
$cursos = $conn->query("SELECT id, nombre FROM curso");
$roles = $conn->query("SELECT id, nombre FROM roles");
// Obtener cursos
$cursos = $conn->query("SELECT id, nombre FROM curso");
// Obtener roles
$roles = $conn->query("SELECT id, nombre FROM roles");
if ($cursos === FALSE) {
    die("Error al obtener los cursos: " . $conn->error);
}
if ($roles === FALSE) {
    die("Error al obtener los roles: " . $conn->error);
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Administrar Usuarios</title>
    <link rel="stylesheet" href="https://fonts.googleapis.com/css?family=Roboto:400,700&display=swap">
    <link rel="stylesheet" href="../css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">

</head>
<body>
    <header>
    <img src="../images/INSIGNIA%20SF.png" alt="Logo de la Empresa">
    <h1>Administrar Usuarios</h1>
    <nav>
        <ul>
            <li><a href="../index.php">Inicio</a></li>
        </ul>
    </nav>
</header>
    <main>
        <?php if ($message): ?>
            <div class="mensaje <?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?>
            </div>
        <?php endif; ?>
        <div class="search-container">
            <form method="GET" action="admin.php">
                <input type="text" name="search" placeholder="Buscar por RUT, nombre, apellidos, curso o rol" value="<?php echo isset($_GET['search']) ? htmlspecialchars($_GET['search'], ENT_QUOTES, 'UTF-8') : ''; ?>">
                <button type="submit" name="buscar">Buscar</button>
            </form>
        </div>
        <form id="formulario-usuario" method="POST" action="admin.php">
            <input type="hidden" name="usuario_id" id="usuario_id">
            <label for="rut">RUT:</label>
            <input type="text" id="rut" name="rut" placeholder="Ejemplo: 12345678-9" required>
            <label for="nombre">Nombre:</label>
            <input type="text" id="nombre" name="nombre" required>
            <label for="apellido_paterno">Apellido Paterno:</label>
            <input type="text" id="apellido_paterno" name="apellido_paterno" required>
            <label for="apellido_materno">Apellido Materno:</label>
            <input type="text" id="apellido_materno" name="apellido_materno" required>
            <label for="contrasena">Contraseña:</label>
            <input type="password" id="contrasena" name="contrasena" pattern=".{8,}" title="La contraseña debe tener al menos 8 caracteres" required>
             <span id="contrasenaHelp" style="font-size: 12px; color: #666;">Ejemplo de contraseña: abc@2024</span>
            <button type="button" id="togglePassword" onclick="togglePasswordVisibility()">Mostrar Contraseña</button>
           
            <label for="rol">Rol:</label>
            <select id="rol" name="rol" required>
                <?php while ($rol = $roles->fetch_assoc()): ?>
                    <option value="<?php echo htmlspecialchars($rol['id'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($rol['nombre'], ENT_QUOTES, 'UTF-8'); ?></option>
                <?php endwhile; ?>
            </select>
            <label for="curso_id">Curso (solo para profesores y estudiantes):</label>
            <select id="curso_id" name="curso_id">
                <option value="">Ninguno</option>
                <?php while ($curso = $cursos->fetch_assoc()): ?>
                    <option value="<?php echo htmlspecialchars($curso['id'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($curso['nombre'], ENT_QUOTES, 'UTF-8'); ?></option>
                <?php endwhile; ?>
            </select>
            <label for="activo">Activo:</label>
            <input type="checkbox" id="activo" name="activo" checked>
            <button type="submit" name="crear" id="crear">Crear Usuario</button>
            <button type="submit" name="modificar" id="modificar" style="display:none;">Modificar Usuario</button>
            <button type="button" onclick="cancelarEdicion()" style="display:none;" id="cancelar">Cancelar</button>
        </form>
        <h2 style="text-align: center;">Descargar Usuarios</h2>
<!-- Botón para exportar a Excel -->
<form method="post" action="exportar_excel.php" style="margin-bottom: 15px;">
    <button type="submit">Exportar a Excel</button>
</form>

        <h2 style="text-align: center;">Usuarios Existentes</h2>
        <form method="POST" action="admin.php">
            <div class="table-responsive">
    <table>
        <thead>
            <tr>
                <th class="checkbox-column">
                    <input type="checkbox" onclick="toggleAllCheckboxes(this)">
                </th>
                <th>RUT</th>
                <th>Nombre Completo</th>
                <th>Rol</th>
                <th>Curso</th>
                <th>Activo</th>
                <th class="actions">Acciones</th>
            </tr>
        </thead>
        <tbody>
            <?php if ($usuarios->num_rows > 0): ?>
                <?php while ($row = $usuarios->fetch_assoc()): ?>
                    <tr class="<?php echo $row['activo'] ? '' : 'inactive'; ?>">
                        <td class="checkbox-column">
                            <input type="checkbox" name="usuario_ids[]" value="<?php echo $row['id']; ?>">
                        </td>
                        <td><?php echo htmlspecialchars($row['rut'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td><?php echo htmlspecialchars($row['nombre'] . ' ' . $row['apellido_paterno'] . ' ' . $row['apellido_materno'], ENT_QUOTES, 'UTF-8'); ?></td>
                        <td>
                            <?php 
                            $rol_result = $conn->query("SELECT nombre FROM roles WHERE id = " . (int)$row['rol_id']);
                            if ($rol_result && $rol_result->num_rows > 0) {
                                $rol = $rol_result->fetch_assoc();
                                echo htmlspecialchars($rol['nombre'], ENT_QUOTES, 'UTF-8');
                            } else {
                                echo 'N/A';
                            }
                            ?>
                        </td>
                        <td>
                            <?php 
                            if (isset($row['curso_id']) && $row['curso_id']) {
                                $curso_result = $conn->query("SELECT nombre FROM curso WHERE id = " . (int)$row['curso_id']);
                                if ($curso_result && $curso_result->num_rows > 0) {
                                    $curso = $curso_result->fetch_assoc();
                                    echo htmlspecialchars($curso['nombre'], ENT_QUOTES, 'UTF-8');
                                } else {
                                    echo 'N/A';
                                }
                            } else {
                                echo 'N/A';
                            }
                            ?>
                        </td>
                        <td><?php echo $row['activo'] ? 'Sí' : 'No'; ?></td>
                        <td class="actions">
                            <button type="button" class="btn btn-edit" onclick="editarUsuario('<?php echo $row['id']; ?>', '<?php echo htmlspecialchars($row['rut'], ENT_QUOTES, 'UTF-8'); ?>', '<?php echo htmlspecialchars($row['nombre'], ENT_QUOTES, 'UTF-8'); ?>', '<?php echo htmlspecialchars($row['apellido_paterno'], ENT_QUOTES, 'UTF-8'); ?>', '<?php echo htmlspecialchars($row['apellido_materno'], ENT_QUOTES, 'UTF-8'); ?>', '<?php echo $row['rol_id']; ?>', '<?php echo $row['curso_id']; ?>', '<?php echo $row['activo']; ?>')">Editar</button>
                            <button type="button" class="btn btn-delete" onclick="eliminarUsuario('<?php echo $row['id']; ?>')">Eliminar</button>
                        </td>
                    </tr>
                <?php endwhile; ?>
            <?php else: ?>
                <tr>
                    <td colspan="7" style="text-align:center;">No se encontraron usuarios.</td>
                </tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>
            <button type="submit" name="eliminar_varios" onclick="return confirm('¿Estás seguro de que quieres eliminar los usuarios seleccionados?');">Eliminar Seleccionados</button>
        </form>
       <!-- Enlaces de paginación -->
<?php if ($total_paginas > 1): ?>
    <div class="pagination" style="text-align: center; margin: 20px 0; flex-wrap: wrap; display: flex; justify-content: center;">
        <?php for ($i = 1; $i <= $total_paginas; $i++): ?>
            <?php if ($i == $pagina_actual): ?>
                <span class="current-page" style="padding: 5px 10px; margin: 0 5px; background-color: #4CAF50; color: white; border-radius: 5px; font-weight: bold;">
                    <?php echo $i; ?>
                </span>
            <?php else: ?>
                <a href="?pagina=<?php echo $i; ?>" style="padding: 5px 10px; margin: 0 5px; text-decoration: none; color: #3498db; border: 1px solid #3498db; border-radius: 5px; transition: background-color 0.3s;">
                    <?php echo $i; ?>
                </a>
            <?php endif; ?>
        <?php endfor; ?>
    </div>
<?php endif; ?>
    </main>
        <script>
        function togglePasswordVisibility() {
            const passwordField = document.getElementById('contrasena');
            const toggleButton = document.getElementById('togglePassword');
            if (passwordField.type === 'password') {
                passwordField.type = 'text';
                toggleButton.textContent = 'Ocultar Contraseña';
            } else {
                passwordField.type = 'password';
                toggleButton.textContent = 'Mostrar Contraseña';
            }
        }
        function editarUsuario(id, rut, nombre, apellido_paterno, apellido_materno, rol_id, curso_id, activo) {
            // Llenar los campos del formulario
            document.getElementById('usuario_id').value = id;
            document.getElementById('rut').value = rut;
            document.getElementById('nombre').value = nombre;
            document.getElementById('apellido_paterno').value = apellido_paterno;
            document.getElementById('apellido_materno').value = apellido_materno;
            document.getElementById('rol').value = rol_id;
            document.getElementById('curso_id').value = curso_id;
            document.getElementById('activo').checked = activo === '1';
            document.getElementById('crear').style.display = 'none';
            document.getElementById('modificar').style.display = 'block';
            document.getElementById('contrasena').required = false;
            document.getElementById('togglePassword').style.display = 'inline-block';
            document.getElementById('contrasenaHelp').style.display = 'block';
            const formulario = document.getElementById('formulario-usuario');
            if (formulario) {
                formulario.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        }
        function cancelarEdicion() {
            document.getElementById('usuario_id').value = '';
            document.getElementById('rut').value = '';
            document.getElementById('nombre').value = '';
            document.getElementById('apellido_paterno').value = '';
            document.getElementById('apellido_materno').value = '';
            document.getElementById('rol').value = '';
            document.getElementById('curso_id').value = '';
            document.getElementById('activo').checked = true;
            document.getElementById('crear').style.display = 'block';
            document.getElementById('modificar').style.display = 'none';
            document.getElementById('contrasena').required = true;
            document.getElementById('togglePassword').style.display = 'inline-block';
            document.getElementById('contrasenaHelp').style.display = 'block';
        }
        function eliminarUsuario(id) {
    if (confirm('¿Estás seguro de que quieres eliminar este usuario permanentemente?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = 'admin.php';
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'usuario_ids[]'; // Para que sea compatible con la eliminación múltiple
        input.value = id;
        const eliminar = document.createElement('input');
        eliminar.type = 'hidden';
        eliminar.name = 'eliminar_varios'; // Activará el bloque correcto en PHP
        eliminar.value = '1';
        form.appendChild(input);
        form.appendChild(eliminar);
        document.body.appendChild(form);
        form.submit();
    }
}
        function toggleAllCheckboxes(source) {
            const checkboxes = document.getElementsByName('usuario_ids[]');
            for (let i = 0; i < checkboxes.length; i++) {
                checkboxes[i].checked = source.checked;
            }
        }
        window.onload = function() {
            const message = document.querySelector('.mensaje');
            if (message) {
                message.classList.add('show');
                setTimeout(() => {
                    message.classList.remove('show');
                }, 2000);
            }
        };
    </script>
</body>
</html>