<?php
session_start(); // Iniciar la sesión

header('Content-Type: text/html; charset=utf-8'); // Establecer la codificación de caracteres

include '../includes/db.php'; // Incluir el archivo de conexión a la base de datos
include '../includes/auth.php'; // Incluir el archivo de autenticación
checkAuth(); // Verificar si el usuario está autenticado

// Verificar si el usuario tiene el rol de profesor o es administrador
if ($_SESSION['rol'] != 'profesor' && !isAdmin()) {
    header("Location: index.php"); // Redirigir a la página de inicio si no tiene permisos
    exit(); // Terminar la ejecución del script
}

// Establecer la codificación de caracteres para la conexión de la base de datos
$conn->set_charset("utf8mb4");

$message = ''; // Variable para almacenar el mensaje

// Función para validar el RUT
function validarRut($rut) {
    $rut = preg_replace('/[^0-9kK]/', '', $rut); // Eliminar puntos y guiones

    if (strlen($rut) < 2) {
        return false;
    }

    $dv = strtoupper(substr($rut, -1)); // Último carácter como dígito verificador
    $numero = substr($rut, 0, strlen($rut) - 1); // Todos menos el último carácter

    $i = 2;
    $suma = 0;
    foreach (array_reverse(str_split($numero)) as $n) {
        if (!is_numeric($n)) {
            return false; // Si algún carácter no es numérico, el RUT no es válido
        }
        $suma += $n * $i;
        $i = ($i == 7) ? 2 : $i + 1;
    }

    $dvr = 11 - ($suma % 11);

    if ($dvr == 11) {
        $dvr = '0';
    }
    if ($dvr == 10) {
        $dvr = 'K';
    }

    return ($dvr == $dv);
}

// Agregar estudiante manualmente
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['crear'])) { // Verificar si se envió el formulario para crear un estudiante
    $rut = $conn->real_escape_string(trim($_POST['rut'])); // Obtener y limpiar el RUT del estudiante
    $nombre = $conn->real_escape_string(trim($_POST['nombre'])); // Obtener y limpiar el nombre del estudiante
    $apellido_paterno = $conn->real_escape_string(trim($_POST['apellido_paterno'])); // Obtener y limpiar el apellido paterno
    $apellido_materno = $conn->real_escape_string(trim($_POST['apellido_materno'])); // Obtener y limpiar el apellido materno
    $curso_id = intval($_POST['curso_id']); // Obtener y convertir el ID del curso a entero
    $contrasena = trim($_POST['contrasena']); // Limpiar la contraseña

    // Verificar si algún campo está vacío o contiene solo espacios en blanco
    if (empty($rut) || empty($nombre) || empty($apellido_paterno) || empty($apellido_materno) || empty($contrasena)) {
        $message = "Error: Todos los campos son obligatorios y no pueden contener solo espacios.";
    } else {
        // Validar el RUT
        if (!validarRut($rut)) {
            $message = "Error: El RUT ingresado no es válido.";
        } else {
            // Verificar si el RUT ya existe en la base de datos
            $stmt = $conn->prepare("SELECT id FROM usuario WHERE rut = ?");
            $stmt->bind_param("s", $rut);
            $stmt->execute();
            $stmt->store_result();

            if ($stmt->num_rows > 0) {
                $message = "Error: El RUT ya está registrado.";
            } else {
                // Validar que la contraseña tenga al menos 8 caracteres
                if (strlen($contrasena) < 8) {
                    $message = "Error: La contraseña debe tener al menos 8 caracteres.";
                } else {
                    $contrasena_hashed = password_hash($contrasena, PASSWORD_BCRYPT); // Hashear la contraseña

                    // Obtener el id del rol 'estudiante' de la tabla roles
                    $rol_query = $conn->prepare("SELECT id FROM roles WHERE nombre = 'estudiante'");
                    $rol_query->execute();
                    $rol_result = $rol_query->get_result();
                    $rol = $rol_result->fetch_assoc();
                    $rol_id = $rol['id']; // Obtener el ID del rol estudiante

                    // Preparar la consulta para insertar un nuevo estudiante
                    $stmt = $conn->prepare("INSERT INTO usuario (nombre, apellido_paterno, apellido_materno, rut, contraseña, rol_id, curso_id) VALUES (?, ?, ?, ?, ?, ?, ?)");
                    $stmt->bind_param("ssssssi", $nombre, $apellido_paterno, $apellido_materno, $rut, $contrasena_hashed, $rol_id, $curso_id); // Vincular los parámetros a la consulta
                    if ($stmt->execute()) { // Ejecutar la consulta y verificar si se realizó con éxito
                        $message = "Alumno agregado correctamente."; // Guardar mensaje de éxito
                    } else {
                        $message = "Error al agregar el alumno."; // Guardar mensaje de error
                    }
                    $stmt->close(); // Cerrar la declaración preparada
                }
            }
        }
    }
}

// Obtener la lista de cursos
$cursos = $conn->query("SELECT id, nombre FROM curso"); // Ejecutar la consulta para obtener todos los cursos
if ($cursos === FALSE) { // Verificar si hubo un error al obtener los cursos
    die("Error al obtener los cursos: " . $conn->error); // Mostrar el error y terminar la ejecución
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Agregar Alumnos</title>
    <link rel="stylesheet" href="https://fonts.googleapis.com/css?family=Roboto:400,700&display=swap">
     <link rel="stylesheet" href="../css/agregar_estudiante.css">
    <style>
        
    </style>
    <script>
        // Función para mostrar un mensaje de notificación
        function showMessage(message) {
            const messageBox = document.getElementById('messageBox');
            messageBox.textContent = message;
            messageBox.style.display = 'block';
            setTimeout(() => {
                messageBox.style.display = 'none';
            }, 3000);
        }

        // Función para mostrar/ocultar la contraseña
        function togglePasswordVisibility() {
            const passwordField = document.getElementById('contrasena');
            const toggleButton = document.getElementById('togglePassword');

            if (passwordField.type === 'password') {
                passwordField.type = 'text';
                toggleButton.textContent = 'Ocultar Contraseña';
            } else {
                passwordField.type = 'password';
                toggleButton.textContent = 'Mostrar Contraseña';
            }
        }

        <?php if ($message): ?> 
        window.onload = function() {
            showMessage('<?php echo $message; ?>');
        }
        <?php endif; ?>
    </script>
</head>
<body>
    <header>
        <img src="../images/INSIGNIA SF.png" alt="Escuela Porongo Insignia">
        <h1>Agregar Alumnos</h1>
    
    <nav>
        <ul>
            <li><a href="../index.php">Inicio</a></li>
        </ul>
    </nav>
        </header>
    <main>
        <?php if ($message): ?>
            <div class="mensaje <?php echo strpos($message, 'Error') === false ? 'exito' : 'error'; ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>

        <!-- Formulario para agregar estudiantes manualmente -->
        <form method="POST" action="add_student.php" onsubmit="return validarFormulario()">
            <label for="rut">RUT (ejemplo: 12345678-9):</label>
            <input type="text" id="rut" name="rut" placeholder="Ejemplo: 12345678-9" required>
            
            <label for="nombre">Nombre del Alumno:</label>
            <input type="text" id="nombre" name="nombre" required>

            <label for="apellido_paterno">Apellido Paterno:</label>
            <input type="text" id="apellido_paterno" name="apellido_paterno" required>

            <label for="apellido_materno">Apellido Materno:</label>
            <input type="text" id="apellido_materno" name="apellido_materno" required>
            
            <label for="contrasena">Contraseña (mínimo 8 caracteres, ejemplo: abc@2024):</label>
            <input type="password" id="contrasena" name="contrasena" pattern=".{8,}" title="La contraseña debe tener al menos 8 caracteres" required>
            <button type="button" id="togglePassword" onclick="togglePasswordVisibility()">Mostrar Contraseña</button> <!-- Botón para mostrar/ocultar contraseña -->

            <label for="curso_id">Curso:</label>
            <select id="curso_id" name="curso_id" required>
                <?php while ($curso = $cursos->fetch_assoc()): ?>
                    <option value="<?php echo htmlspecialchars($curso['id'], ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($curso['nombre'], ENT_QUOTES, 'UTF-8'); ?></option>
                <?php endwhile; ?>
            </select>

            <button type="submit" name="crear">Agregar Alumno</button>
        </form>

        <p><a href="import_excel.php" class="button">Importar estudiantes desde archivo Excel</a></p>
    </main>
    <div id="messageBox" class="mensaje"></div> <!-- Caja para mostrar mensajes -->

    <script>
        function validarFormulario() {
            const rut = document.getElementById('rut').value.trim();
            const nombre = document.getElementById('nombre').value.trim();
            const apellidoPaterno = document.getElementById('apellido_paterno').value.trim();
            const apellidoMaterno = document.getElementById('apellido_materno').value.trim();
            const contrasena = document.getElementById('contrasena').value.trim();

            if (!rut || !nombre || !apellidoPaterno || !apellidoMaterno || !contrasena) {
                alert('Error: Todos los campos son obligatorios y no pueden estar vacíos o contener solo espacios en blanco.');
                return false;
            }
            return true;
        }
    </script>
</body>
</html>
