<?php
session_start();  // Iniciar la sesión

include '../includes/db.php'; // Incluir archivo de conexión a la base de datos
include '../includes/auth.php'; // Incluir archivo de autenticación
checkAuth(); // Verificar si el usuario está autenticado

// Establecer la codificación de caracteres para la conexión de la base de datos
$conn->set_charset("utf8");

$message = ''; // Variable para almacenar el mensaje
$message_type = 'success'; // Tipo de mensaje por defecto

// Paginación
$limite = 20; // Número de registros por página
$pagina = isset($_GET['pagina']) ? (int)$_GET['pagina'] : 1;
$offset = ($pagina - 1) * $limite;

// Lógica para agregar un usuario desde la información de la matrícula
if (isset($_GET['add_user_id'])) {
    $matricula_id = intval($_GET['add_user_id']);

    // Obtener la información de la matrícula
    $sql = "SELECT rut, nombre, apellido_paterno, apellido_materno, curso_id FROM matricula WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $matricula_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $matricula = $result->fetch_assoc();

    if ($matricula) {
        // Verificar si el RUT ya está registrado en la tabla 'usuario'
        $sql_check_rut = "SELECT id FROM usuario WHERE rut = ?";
        $stmt_check = $conn->prepare($sql_check_rut);
        $stmt_check->bind_param("s", $matricula['rut']);
        $stmt_check->execute();
        $stmt_check->store_result();

        if ($stmt_check->num_rows > 0) {
            // El RUT ya está registrado en el sistema
            $message = "Error: El RUT ya está registrado en el sistema.";
            $message_type = 'error';
        } else {
            // Establecer el rol por defecto como 'estudiante' (suponiendo que el rol 'estudiante' tiene el rol_id = 3)
            $rol_id = 3;
            $contraseña = password_hash($matricula['rut'], PASSWORD_BCRYPT); // Usar el RUT como contraseña

            // Insertar el usuario en la tabla 'usuario'
            $sql_insert = "INSERT INTO usuario (rut, nombre, apellido_paterno, apellido_materno, contraseña, rol_id, curso_id) VALUES (?, ?, ?, ?, ?, ?, ?)";
            $stmt_insert = $conn->prepare($sql_insert);
            $stmt_insert->bind_param(
                "ssssssi",
                $matricula['rut'],
                $matricula['nombre'],
                $matricula['apellido_paterno'],
                $matricula['apellido_materno'],
                $contraseña,
                $rol_id,
                $matricula['curso_id']
            );

            if ($stmt_insert->execute()) {
                $message = "Usuario creado exitosamente.";
            } else {
                $message = "Error al crear el usuario: " . $stmt_insert->error;
                $message_type = 'error';
            }
        }
    } else {
        $message = "Matrícula no encontrada.";
        $message_type = 'error';
    }
}

// Lógica para eliminar matrícula y datos relacionados
if (isset($_GET['delete_id'])) {
    $matricula_id = intval($_GET['delete_id']);

    // Eliminar registros relacionados con la matrícula
    $conn->begin_transaction(); // Iniciar una transacción para asegurar la consistencia de los datos

    try {
        // Eliminar registros en las tablas relacionadas
        $conn->query("DELETE FROM usuario WHERE curso_id IN (SELECT curso_id FROM matricula WHERE id = $matricula_id)");
        $conn->query("DELETE FROM previsiones_salud WHERE matricula_id = $matricula_id");
        $conn->query("DELETE FROM informacion_adicional WHERE matricula_id = $matricula_id");
        $conn->query("DELETE FROM familiares WHERE matricula_id = $matricula_id");
        $conn->query("DELETE FROM apoderados WHERE matricula_id = $matricula_id");
        $conn->query("DELETE FROM alergias_alimentarias WHERE matricula_id = $matricula_id");

        // Finalmente, eliminar la matrícula
        $conn->query("DELETE FROM matricula WHERE id = $matricula_id");

        // Confirmar la transacción
        $conn->commit();
        $message = "Matrícula eliminada exitosamente.";
    } catch (Exception $e) {
        $conn->rollback(); // Deshacer la transacción en caso de error
        $message = "Error al eliminar la matrícula: " . $e->getMessage();
        $message_type = 'error';
    }
}

// Lógica para buscar matrículas por RUT
$search_rut = '';
if (isset($_POST['search_rut'])) {
    $search_rut = $_POST['search_rut'];
}

// Obtener matrículas junto con el nombre del curso de la base de datos
$matriculas = $conn->query("
    SELECT matricula.*, curso.nombre AS curso_nombre, matricula.rut, matricula.nombre AS nombre_alumno, matricula.apellido_paterno, matricula.apellido_materno 
    FROM matricula 
    JOIN curso ON matricula.curso_id = curso.id
    " . ($search_rut ? "WHERE matricula.rut LIKE '%" . $conn->real_escape_string($search_rut) . "%'" : "") . 
    " LIMIT $offset, $limite"
);

// Obtener el número total de registros para la paginación
$total_matriculas = $conn->query("SELECT COUNT(*) as total FROM matricula")->fetch_assoc()['total'];
$total_paginas = ceil($total_matriculas / $limite);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestionar Matrículas</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="../css/matricula_crud.css">
    <style>
       
    </style>
    <script>
        function showMessage(message, type) {
            const messageBox = document.getElementById('messageBox');
            messageBox.textContent = message;
            messageBox.style.backgroundColor = type === 'success' ? '#4CAF50' : '#f44336';
            messageBox.style.display = 'block';
            setTimeout(() => {
                messageBox.style.display = 'none';
            }, 3000);
        }

        <?php if ($message): ?> 
        window.onload = function() {
            showMessage('<?php echo $message; ?>', '<?php echo $message_type; ?>');
        }
        <?php endif; ?>
    </script>
</head>
<body>
    <header>
        <img src="../images/INSIGNIA SF.png" alt="Escuela Porongo Insignia">
        <h1>Gestionar Matrículas</h1>
    
    <nav>
        <ul>
            <li><a href="../index.php">Inicio</a></li>
            <li><a href="formulario_matricula.php">Agregar Matrícula</a></li>
        </ul>
    </nav>
        </header>
    <main>
        <div class="search-container">
            <form method="POST" action="">
                <input type="text" name="search_rut" placeholder="Buscar por RUT" value="<?php echo htmlspecialchars($search_rut); ?>">
                <button type="submit">Buscar</button>
            </form>
        </div>
        <table>
            <thead>
                <tr>
                    <th>RUT</th>
                    <th>Apellido Paterno</th>
                    <th>Apellido Materno</th>
                    <th>Nombre</th>
                    <th>Curso</th>
                    <th>Acciones</th>
                </tr>
            </thead>
            <tbody>
                <?php while ($row = $matriculas->fetch_assoc()): ?>
                <tr>
                    <td><?php echo htmlspecialchars($row['rut'] ?? ''); ?></td>
                    <td><?php echo htmlspecialchars($row['apellido_paterno'] ?? ''); ?></td>
                    <td><?php echo htmlspecialchars($row['apellido_materno'] ?? ''); ?></td>
                    <td><?php echo htmlspecialchars($row['nombre_alumno'] ?? ''); ?></td>
                    <td><?php echo htmlspecialchars($row['curso_nombre'] ?? ''); ?></td>
                    <td class="actions">
                        <a href="ver_matricula.php?id=<?php echo $row['id']; ?>" class="view">Ver</a>
                        <a href="editar_matricula.php?id=<?php echo $row['id']; ?>" class="edit">Editar</a>
                        <a href="?delete_id=<?php echo $row['id']; ?>" class="delete" onclick="return confirm('¿Está seguro de que desea eliminar esta matrícula y todos los datos relacionados?')">Eliminar</a>
                        <a href="?add_user_id=<?php echo $row['id']; ?>" class="add-user">Agregar Usuario</a>
                    </td>
                </tr>
                <?php endwhile; ?>
            </tbody>
        </table>

        <div class="pagination">
            <?php for ($i = 1; $i <= $total_paginas; $i++): ?>
                <a href="?pagina=<?php echo $i; ?>" class="<?php echo ($i == $pagina) ? 'active' : ''; ?>"><?php echo $i; ?></a>
            <?php endfor; ?>
        </div>
    </main>
    <div id="messageBox" class="message"></div>
</body>
</html>
